/*
    Copyright (C) 2016 Matthias P. Braendli

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <Adafruit_LiquidCrystal.h>
#include <RotaryEncoder.h>
#include <ezButton.h>

const int encPinA = A2;
const int encPinB = A1;
const int encPinBtn = A3;
static int encoderPos = 0;
RotaryEncoder encoder(encPinA, encPinB, RotaryEncoder::LatchMode::TWO03);
ezButton encButton(encPinBtn);


const int modeSwitchA = 4;
const int modeSwitchB = 6;

const int pwmRedPin = 11;
const int pwmGreenPin = 9;
const int pwmBluePin = 10;

constexpr int MAX_VALUE = 4;
static int valueRed = 4;
static int valueGreen = 3;
static int valueBlue = 3;

// 2000 * math.pow(2, n/3)
constexpr int N_INTERVALS = 16;
const static unsigned long fStop1_3Durations[16] = {
 2000, 2520, 3175,
 4000, 5040, 6350,
 8000, 10079, 12699,
 16000, 20159, 25398,
 32000, 40317, 50797,
 64000};


static int timeSetting = 8; // index of fStop1_3Durations table

enum class CountingMode { NORMAL, TESTSTRIP_WIDE, TESTSTRIP_F1_3 };

/* test strip with cumulative exposure times
 * 5 7.1 10 14.1 20 28.3 40 seconds
 *
 * This is 5s base and then half-stop increments.
 *
 * steps: 5      2.1       2.9         4.1 5.9 8.3 11.7
 *      _____    _____    _________
 * _____|   |____|   |____|       | etc.
 *
 * step: 0   1    2   3    4           6   8   10  12
 *
 * Always 2s pauses. Odd steps are pauses.
 */
static CountingMode currentMode = CountingMode::NORMAL;
static unsigned long countStartedAt = 0;

constexpr int TESTSTRIP_WIDE_STEPS = 13;
const static unsigned long testripDurations[7] = { 5000, 2100, 2900, 4100, 5900, 8300, 11700 };

constexpr int PAUSE_DUR_MS = 2500;
static int testStripStep = 0;
static unsigned long nextToggleAt = 0;

/* f 1/3 stop around nominal exposure time test strip:
 *
 * cumulative illumination time:
 *      n-f2/3    n-f1/3      n        n+f1/3     n+f2/3
 *      _____     _____     ________
 * _____|   |_____|   |_____|      |___ etc.
 *
 * Always 2s pauses, and illumination durations so that cumulatively
 * it adds up to the five desired f-stops.
 *
 * Odd steps are pauses.
 */
constexpr int TESTSTRIP_F1_3_STEPS = 9;

// Cube root of 2 is quite close to 63/50, and its square close to 79/50
// which allows us to calculate only with integer math.
constexpr int CUBE_ROOT_2_NOM = 63;
constexpr int CUBE_ROOT_2_DENOM = 50;
constexpr int CUBE_ROOT_4_NOM = 79;
constexpr int CUBE_ROOT_4_DENOM = 50;
unsigned long teststrip_f1_3_step_duration = 0;
unsigned long teststrip_f1_3_cumulated_time = 0;

enum class State { RED, GREEN, BLUE, TIME, COUNTING };
static State currentState = State::TIME;

// To distinguish between long and short push
static long encoderButtonPressedTime = 0;
static long encoderButtonReleasedTime = 0;
enum class ButtonPress { NO, SHORT, LONG };

// Uses default I2C port marked SCL SDA on Itsy Bitsy
Adafruit_LiquidCrystal lcd(0);

static State advanceState()
{
  State n;
  switch (currentState) {
    case State::RED: n = State::GREEN; break;
    case State::GREEN: n = State::BLUE; break;
    case State::BLUE: n = State::TIME; break;
    case State::TIME: n = State::RED; break;
    case State::COUNTING: n = State::TIME; break;
  }
  return n;
}

// Scale in powers of two, but low values are actually quite useless
inline int scalePwm(int value)
{
  if (value == 0) return 0;
  else if (value == 1) return 32;
  else if (value == 2) return 64;
  else if (value == 3) return 128;
  else return 255;
}

static void illuminate_paper(bool enable)
{
  if (enable) {
    analogWrite(pwmGreenPin, scalePwm(valueGreen));
    analogWrite(pwmBluePin, scalePwm(valueBlue));
  }
  else {
    analogWrite(pwmGreenPin, 0);
    analogWrite(pwmBluePin, 0);
  }
}

// Doesn't terminate buf, because we anyway need to print trailing spaces
// to clear the LCD line.
static void printDeciSeconds(char* buf, unsigned long milliseconds) {
  // It's faster to do this than calculating
  snprintf(buf, 4, "%03lu", milliseconds/100);
  buf[3] = buf[2];
  buf[2] = '.';
}

static void disp() {
  lcd.setCursor(0, 0);
  if (currentState == State::RED)
    lcd.print(">R");
  else
    lcd.print(" R");

  lcd.print(valueRed);

  if (currentState == State::GREEN)
    lcd.print(" >G");
  else
    lcd.print("  G");

  lcd.print(valueGreen);

  if (currentState == State::BLUE)
    lcd.print(" >B");
  else
    lcd.print("  B");

  lcd.print(valueBlue);
  lcd.print("  ");

  lcd.setCursor(0, 1);
  switch (currentMode) {
    case CountingMode::NORMAL:
      lcd.print("NORM ");
      break;
    case CountingMode::TESTSTRIP_WIDE:
      lcd.print("STRP ");
      lcd.print((testStripStep % 2) == 1 ? "x " : ". ");
      break;
    case CountingMode::TESTSTRIP_F1_3:
      lcd.print("f1/3 ");
      lcd.print((testStripStep % 2) == 1 ? "x " : ". ");
      break;
  }

  constexpr int CHARS_REMAINING = 11;
  char buf[CHARS_REMAINING + 1];
  memset(buf, ' ', CHARS_REMAINING + 1);

  switch (currentMode) {
      case CountingMode::NORMAL:
        if (currentState == State::COUNTING) {
          auto now = millis();
          unsigned long remaining = 0;
          auto endMillis = countStartedAt + fStop1_3Durations[timeSetting];
          if (endMillis > now) remaining = endMillis - now;
          printDeciSeconds(buf, remaining);
        }
        else if (currentState == State::TIME) {
          lcd.print(">T");
          int written = snprintf(buf, CHARS_REMAINING+1, " %2lu s", fStop1_3Durations[timeSetting] / 1000);
          if (written < CHARS_REMAINING + 1) buf[written] = ' ';
        }
        else {
          lcd.print(" T");
          int written = snprintf(buf, CHARS_REMAINING+1, " %2lu s", fStop1_3Durations[timeSetting] / 1000);
          if (written < CHARS_REMAINING + 1) buf[written] = ' ';
        }
        break;

      case CountingMode::TESTSTRIP_WIDE:
        if (currentState == State::COUNTING) {
          auto now = millis();
          unsigned long remaining = 0;
          if (nextToggleAt > now) remaining = nextToggleAt - now;
          snprintf(buf, CHARS_REMAINING, "%2d ", TESTSTRIP_WIDE_STEPS - testStripStep);
          printDeciSeconds(buf + 3, remaining);
          //int written = snprintf(buf, CHARS_REMAINING, "%d/%d", testStripStep, TESTSTRIP_WIDE_STEPS);
          //if (written < CHARS_REMAINING + 1) buf[written] = ' ';
        }
        else {
          int written = snprintf(buf, CHARS_REMAINING+1, "%d steps", TESTSTRIP_WIDE_STEPS);
          if (written < CHARS_REMAINING + 1) buf[written] = ' ';
        }
        break;

      case CountingMode::TESTSTRIP_F1_3:
        if (currentState == State::COUNTING) {
          auto now = millis();
          unsigned long remaining = 0;
          if (nextToggleAt > now) remaining = nextToggleAt - now;
          snprintf(buf, CHARS_REMAINING, "%2d ", TESTSTRIP_F1_3_STEPS - testStripStep);
          printDeciSeconds(buf + 3, remaining);
        }
        else if (currentState == State::TIME) {
          lcd.print(">T");
          snprintf(buf, CHARS_REMAINING+1, " %2lu s  ", fStop1_3Durations[timeSetting] / 1000);
        }
        else {
          lcd.print(" T");
          snprintf(buf, CHARS_REMAINING+1, " %2lu s  ", fStop1_3Durations[timeSetting] / 1000);
        }
        break;
  }

  buf[CHARS_REMAINING] = '\0';
  lcd.print(buf);
}

static CountingMode getCountingMode() {
  if (digitalRead(modeSwitchA) == 0) {
    return CountingMode::TESTSTRIP_WIDE;
  }
  else if (digitalRead(modeSwitchB) == 0) {
    return CountingMode::TESTSTRIP_F1_3;
  }
  else {
    return CountingMode::NORMAL;
  }
}

void setup() {
  if (!lcd.begin(16, 2)) while (1);
  lcd.clear();
  lcd.setBacklight(HIGH);

  encButton.setDebounceTime(50);

  pinMode(pwmRedPin, OUTPUT);
  analogWrite(pwmRedPin, scalePwm(valueRed));
  pinMode(pwmGreenPin, OUTPUT);
  analogWrite(pwmGreenPin, 0);
  pinMode(pwmBluePin, OUTPUT);
  analogWrite(pwmBluePin, 0);

  pinMode(encPinBtn, INPUT_PULLUP);
  pinMode(modeSwitchA, INPUT_PULLUP);
  pinMode(modeSwitchB, INPUT_PULLUP);

  lcd.setCursor(0, 0);
  disp();
}

void loop() {
  const auto now = millis();

  encButton.loop();
  encoder.tick();

  const int newPos = encoder.getPosition();

  int delta = 0;
  if (encoderPos != newPos) {
    delta = newPos - encoderPos;
    encoderPos = newPos;
  }

  ButtonPress button = ButtonPress::NO;
  if (encButton.isPressed()) encoderButtonPressedTime = now;
  if (encButton.isReleased()) {
    encoderButtonReleasedTime = now;
    const long pressDuration = encoderButtonReleasedTime - encoderButtonPressedTime;

    if (pressDuration < 500) {
      button = ButtonPress::SHORT;
    }
    else {
      button = ButtonPress::LONG;
    }
  }

  bool updateDisp = false;

  auto nextState = currentState;

  if (currentState != State::COUNTING) {
    auto mode = getCountingMode();
    if (currentMode != mode) {
      currentMode = mode;
      updateDisp = true;
    }
  }

  if (currentState == State::TIME) {
    // short press => advance state
    // long press => start timer

    switch (button) {
      case ButtonPress::SHORT:
        nextState = advanceState();
        updateDisp = true;
        break;
      case ButtonPress::LONG:
        nextState = State::COUNTING;
        countStartedAt = now;
        nextToggleAt = countStartedAt;
        testStripStep = 0;
        updateDisp = true;
        break;
      case ButtonPress::NO: break;
    }
  }
  else if (currentState == State::COUNTING) {
    bool done = false;

    switch (currentMode) {
      case CountingMode::NORMAL:
        done = (countStartedAt + fStop1_3Durations[timeSetting]) < now;
        illuminate_paper(true);
        break;

      case CountingMode::TESTSTRIP_WIDE:
        if (nextToggleAt < now) {
          if (testStripStep % 2 == 0) {
              nextToggleAt += testripDurations[testStripStep/2];
              illuminate_paper(true);
          }
          else {
              nextToggleAt += PAUSE_DUR_MS;
              illuminate_paper(false);
          }
          testStripStep++;
        }
        done = testStripStep == TESTSTRIP_WIDE_STEPS + 1;
        break;

      case CountingMode::TESTSTRIP_F1_3:
        if (nextToggleAt < now) {
          switch (testStripStep) {
            case 1:
            case 3:
            case 5:
            case 7:
              nextToggleAt += PAUSE_DUR_MS;
              illuminate_paper(false);
              break;

            case 0:
              teststrip_f1_3_cumulated_time = 0;
              teststrip_f1_3_step_duration = fStop1_3Durations[timeSetting] * CUBE_ROOT_4_DENOM / CUBE_ROOT_4_NOM;
              nextToggleAt += teststrip_f1_3_step_duration;
              illuminate_paper(true);
              break;
            case 2:
              teststrip_f1_3_cumulated_time += teststrip_f1_3_step_duration;
              teststrip_f1_3_step_duration =
                (fStop1_3Durations[timeSetting] * CUBE_ROOT_2_DENOM / CUBE_ROOT_2_NOM) -
                teststrip_f1_3_cumulated_time;

              nextToggleAt += teststrip_f1_3_step_duration;
              illuminate_paper(true);
              break;
            case 4:
              teststrip_f1_3_cumulated_time += teststrip_f1_3_step_duration;
              teststrip_f1_3_step_duration = fStop1_3Durations[timeSetting] - teststrip_f1_3_cumulated_time;
              nextToggleAt += teststrip_f1_3_step_duration;
              illuminate_paper(true);
              break;
            case 6:
              teststrip_f1_3_cumulated_time += teststrip_f1_3_step_duration;
              teststrip_f1_3_step_duration = (fStop1_3Durations[timeSetting] * CUBE_ROOT_2_NOM / CUBE_ROOT_2_DENOM) -
                teststrip_f1_3_cumulated_time;

              nextToggleAt += teststrip_f1_3_step_duration;
              illuminate_paper(true);
              break;
            case 8:
              teststrip_f1_3_cumulated_time += teststrip_f1_3_step_duration;
              teststrip_f1_3_step_duration = (fStop1_3Durations[timeSetting] * CUBE_ROOT_4_NOM / CUBE_ROOT_4_DENOM) -
                teststrip_f1_3_cumulated_time;

              nextToggleAt += teststrip_f1_3_step_duration;
              illuminate_paper(true);
              break;
          }
          testStripStep++;
        }

        done = testStripStep == TESTSTRIP_F1_3_STEPS + 1;
        break;
    }

    if (button != ButtonPress::NO || done) {
      nextState = State::TIME;
      testStripStep = 0;
      illuminate_paper(false);
    }
    updateDisp = true;
  }
  else {
    if (button == ButtonPress::SHORT) {
      nextState = advanceState();
      updateDisp = true;
    }
  }

  if (currentState != State::COUNTING) {
    if (delta) {
      switch (currentState) {
        case State::RED:
          valueRed += delta;
          if (valueRed < 0) valueRed = 0;
          if (valueRed > MAX_VALUE) valueRed = MAX_VALUE;
          break;

        case State::GREEN:
          valueGreen += delta;
          if (valueGreen < 0) valueGreen = 0;
          if (valueGreen > MAX_VALUE) valueGreen = MAX_VALUE;
          break;

        case State::BLUE:
          valueBlue += delta;
          if (valueBlue < 0) valueBlue = 0;
          if (valueBlue > MAX_VALUE) valueBlue = MAX_VALUE;
          break;

        case State::TIME:
          timeSetting += delta;
          if (timeSetting < 0) timeSetting = 0;
          if (timeSetting >= N_INTERVALS) timeSetting = N_INTERVALS-1;
          break;

        case State::COUNTING: break;
      }

      if (currentState == State::RED) {
        analogWrite(pwmRedPin, scalePwm(valueRed));
      }

      updateDisp = true;
    }
  }

  currentState = nextState;
  if (updateDisp) disp();
}
